# SPDX-FileCopyrightText: © 2004 Python Software Foundation. <https://www.python.org/psf/>
#
# SPDX-License-Identifier: Python-2.0
# All rights reserved.

# pylint: disable=redefined-builtin,too-many-arguments,too-many-locals

from jepler_udecimal import Decimal


def moneyfmt(
    value, *, places=2, curr="", sep=",", dp=".", pos="", neg="-", trailneg=""
):
    """Convert Decimal to a money formatted string.

    places:  required number of places after the decimal point
    curr:    optional currency symbol before the sign (may be blank)
    sep:     optional grouping separator (comma, period, space, or blank)
    dp:      decimal point indicator (comma or period)
             only specify as blank when places is zero
    pos:     optional sign for positive numbers: '+', space or blank
    neg:     optional sign for negative numbers: '-', '(', space or blank
    trailneg:optional trailing minus indicator:  '-', ')', space or blank

    >>> d = Decimal('-1234567.8901')
    >>> moneyfmt(d, curr='$')
    '-$1,234,567.89'
    >>> moneyfmt(d, places=0, sep='.', dp='', neg='', trailneg='-')
    '1.234.568-'
    >>> moneyfmt(d, curr='$', neg='(', trailneg=')')
    '($1,234,567.89)'
    >>> moneyfmt(Decimal(123456789), sep=' ')
    '123 456 789.00'
    >>> moneyfmt(Decimal('-0.02'), neg='<', trailneg='>')
    '<0.02>'

    """
    q = Decimal(10) ** -places  # 2 places --> '0.01'
    sign, digits, _ = value.quantize(q).as_tuple()
    result = []
    digits = list(map(str, digits))
    build, next = result.append, digits.pop
    if sign:
        build(trailneg)
    for i in range(places):
        build(next() if digits else "0")
    if places:
        build(dp)
    if not digits:
        build("0")
    i = 0
    while digits:
        build(next())
        i += 1
        if i == 3 and digits:
            i = 0
            build(sep)
    build(curr)
    build(neg if sign else pos)
    return "".join(reversed(result))


if __name__ == "__main__":
    d = Decimal("-1234567.8901")
    print(moneyfmt(d, curr="$"))
    print(moneyfmt(d, places=0, sep=".", dp="", neg="", trailneg="-"))
    print(moneyfmt(d, curr="$", neg="(", trailneg=")"))
    print(moneyfmt(Decimal(123456789), sep=" "))
    print(moneyfmt(Decimal("-0.02"), neg="<", trailneg=">"))
